/** @file   parametertable.h
 * @brief   Declaration of ParameterTable - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_PARAMETERTABLE_H
#define H_ENG2D_PARAMETERTABLE_H

#include <map>
#include <string>
#include "eng2d_dll.h"

namespace eng2d {


/** @class  ParameterTable
 * @brief   Represents a table that can store parameters.
 * @author  Tomi Lamminsaari
 *
 */
class DLLIMPORT ParameterTable
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================



  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	ParameterTable();


	/** Destructor
   */
	virtual ~ParameterTable();

	/** Copy constructor.
   * @param     rO                Reference to another ParameterTable
   */
  ParameterTable( const ParameterTable& rO );

	/** Assignment operator
   * @param     rO                Reference to another ParameterTable
   * @return    Reference to us.
   */
  ParameterTable& operator = ( const ParameterTable& rO );



  ///
  /// Methods
  /// =======

  /** Stores the given parameterpair to this table.
   * @param     parKey            A string that operates as a key - value.
   * @param     parValue          A string that contains the value of the
   *                              parameter.
   */
  void addParameter( const std::string& parKey, const std::string& parValue );


  ///
  /// Getter methods
  /// ==============

  /** Returns the value of requested parameter.
   * @param     parKey            The key that unifies the parameter.
   * @return    A string containing the value or an empty string if there is
   *            no match.
   */
  std::string get( const std::string& parKey ) const;
  
  /** Returns the value of requested parameter as an integer.
   * @param     parKey            The key that unifies the parameter.
   * @return    An integer containing the value. If there is no parameter that
   *            matches the key, a zero will be returned.
   */
  int getInt( const std::string& parKey ) const;
  
  /** Returns the value of requested parameter as a floating point number.
   * @param     parKey            The key that unifies the parameter.
   * @return    A floating point number.
   */
  float getFloat( const std::string& parKey ) const;
  
  /** Returns the number of parameters there are.
   * @return    Number of parameters this table contains.
   */
  int size() const;

  /** Tells if there exists a parameter with given name.
   * @param     parKey            A string containing the name of the parameter.
   * @return    true if that parameter exists.
   */
  bool exists( const std::string& parKey ) const;
  
  
protected:

  ///
  /// Members
  /// =======
  
  /** The parameters are stored here. */
  std::map< std::string, std::string >  m_parameters;



private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif
